<?php

declare(strict_types=1);

namespace Laminas\Json\Server;

use Laminas\Json\Json;
use Laminas\Json\Server\Exception\InvalidArgumentException;
use Laminas\Json\Server\Exception\RuntimeException;

use function array_key_exists;
use function assert;
use function in_array;
use function is_array;
use function is_string;
use function method_exists;
use function preg_match;
use function ucfirst;

class Smd
{
    public const ENV_JSONRPC_1 = 'JSON-RPC-1.0';
    public const ENV_JSONRPC_2 = 'JSON-RPC-2.0';
    public const SMD_VERSION   = '2.0';

    /**
     * Content type.
     *
     * @var string
     */
    protected $contentType = 'application/json';

    /**
     * Content type regex.
     *
     * @var string
     */
    protected $contentTypeRegex = '#[a-z]+/[a-z][a-z-]+#i';

    /**
     * Service description.
     *
     * @var string
     */
    protected $description;

    /**
     * Generate Dojo-compatible SMD?
     *
     * @var bool
     */
    protected $dojoCompatible = false;

    /**
     * Current envelope.
     *
     * @var string
     */
    protected $envelope = self::ENV_JSONRPC_1;

    /**
     * Allowed envelope types.
     *
     * @var array
     */
    protected $envelopeTypes = [
        self::ENV_JSONRPC_1,
        self::ENV_JSONRPC_2,
    ];

    /**
     * Service id.
     *
     * @var string
     */
    protected $id;

    /**
     * Services offered.
     *
     * @var array
     */
    protected $services = [];

    /**
     * Service target.
     *
     * @var string
     */
    protected $target;

    /**
     * Global transport.
     *
     * @var string
     */
    protected $transport = 'POST';

    /**
     * Allowed transport types.
     *
     * @var array
     */
    protected $transportTypes = ['POST'];

    /**
     * Set object state via options.
     *
     * @param  array $options
     * @return self
     */
    public function setOptions(array $options)
    {
        foreach ($options as $key => $value) {
            $method = 'set' . ucfirst($key);

            if (method_exists($this, $method)) {
                $this->$method($value);
            }
        }

        return $this;
    }

    /**
     * Set transport.
     *
     * @param  string $transport
     * @return self
     * @throws InvalidArgumentException
     */
    public function setTransport($transport)
    {
        if (! in_array($transport, $this->transportTypes)) {
            throw new InvalidArgumentException("Invalid transport '{$transport}' specified");
        }

        $this->transport = $transport;
        return $this;
    }

    /**
     * Get transport.
     *
     * @return string
     */
    public function getTransport()
    {
        return $this->transport;
    }

    /**
     * Set envelope.
     *
     * @param  string $envelopeType
     * @return self
     * @throws InvalidArgumentException
     */
    public function setEnvelope($envelopeType)
    {
        if (! in_array($envelopeType, $this->envelopeTypes)) {
            throw new InvalidArgumentException("Invalid envelope type '{$envelopeType}'");
        }

        $this->envelope = $envelopeType;
        return $this;
    }

    /**
     * Retrieve envelope.
     *
     * @return string
     */
    public function getEnvelope()
    {
        return $this->envelope;
    }

    /**
     * Set content type
     *
     * @param  string $type
     * @return self
     * @throws InvalidArgumentException
     */
    public function setContentType($type)
    {
        if (! preg_match($this->contentTypeRegex, $type)) {
            throw new InvalidArgumentException("Invalid content type '{$type}' specified");
        }

        $this->contentType = $type;
        return $this;
    }

    /**
     * Retrieve content type
     *
     * Content-Type of response; default to application/json.
     *
     * @return string
     */
    public function getContentType()
    {
        return $this->contentType;
    }

    /**
     * Set service target.
     *
     * @param  string $target
     * @return self
     */
    public function setTarget($target)
    {
        $this->target = (string) $target;
        return $this;
    }

    /**
     * Retrieve service target.
     *
     * @return string
     */
    public function getTarget()
    {
        return $this->target;
    }

    /**
     * Set service ID.
     *
     * @param  string $id
     * @return self
     */
    public function setId($id)
    {
        $this->id = (string) $id;
        return $this;
    }

    /**
     * Get service id.
     *
     * @return string
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * Set service description.
     *
     * @param  string $description
     * @return self
     */
    public function setDescription($description)
    {
        $this->description = (string) $description;
        return $this;
    }

    /**
     * Get service description.
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->description;
    }

    /**
     * Indicate whether or not to generate Dojo-compatible SMD.
     *
     * @param  bool $flag
     * @return self
     */
    public function setDojoCompatible($flag)
    {
        $this->dojoCompatible = (bool) $flag;
        return $this;
    }

    /**
     * Is this a Dojo compatible SMD?
     *
     * @return bool
     */
    public function isDojoCompatible()
    {
        return $this->dojoCompatible;
    }

    /**
     * Add Service.
     *
     * @param Smd\Service|array $service
     * @return self
     * @throws RuntimeException
     * @throws InvalidArgumentException
     */
    public function addService($service)
    {
        if (is_array($service)) {
            $service = new Smd\Service($service);
        }

        if (! $service instanceof Smd\Service) {
            throw new InvalidArgumentException('Invalid service passed to addService()');
        }

        $name = $service->getName();

        if (array_key_exists($name, $this->services)) {
            throw new RuntimeException('Attempt to register a service already registered detected');
        }

        $this->services[$name] = $service;
        return $this;
    }

    /**
     * Add many services.
     *
     * @param  array $services
     * @return self
     */
    public function addServices(array $services)
    {
        foreach ($services as $service) {
            $this->addService($service);
        }

        return $this;
    }

    /**
     * Overwrite existing services with new ones.
     *
     * @param  array $services
     * @return self
     */
    public function setServices(array $services)
    {
        $this->services = [];
        return $this->addServices($services);
    }

    /**
     * Get service object.
     *
     * @param  string $name
     * @return bool|Smd\Service
     */
    public function getService($name)
    {
        if (! array_key_exists($name, $this->services)) {
            return false;
        }

        return $this->services[$name];
    }

    /**
     * Return services.
     *
     * @return array
     */
    public function getServices()
    {
        return $this->services;
    }

    /**
     * Remove service.
     *
     * @param  string $name
     * @return bool
     */
    public function removeService($name)
    {
        if (! array_key_exists($name, $this->services)) {
            return false;
        }

        unset($this->services[$name]);
        return true;
    }

    /**
     * Cast to array.
     *
     * @return array
     */
    public function toArray()
    {
        if ($this->isDojoCompatible()) {
            return $this->toDojoArray();
        }

        $description = $this->getDescription();
        $transport   = $this->getTransport();
        $envelope    = $this->getEnvelope();
        $contentType = $this->getContentType();
        $smdVersion  = static::SMD_VERSION;
        assert(is_string($smdVersion));
        $service = [
            'transport'   => $transport,
            'envelope'    => $envelope,
            'contentType' => $contentType,
            'SMDVersion'  => $smdVersion,
            'description' => $description,
        ];

        if (null !== ($target = $this->getTarget())) {
            $service['target'] = $target;
        }
        if (null !== ($id = $this->getId())) {
            $service['id'] = $id;
        }

        $services = $this->getServices();
        if (empty($services)) {
            return $service;
        }

        $service['services'] = [];
        foreach ($services as $name => $svc) {
            $svc->setEnvelope($envelope);
            $service['services'][$name] = $svc->toArray();
        }
        $service['methods'] = $service['services'];

        return $service;
    }

    /**
     * Export to DOJO-compatible SMD array
     *
     * @return array
     */
    public function toDojoArray()
    {
        $smdVersion  = '.1';
        $serviceType = 'JSON-RPC';
        $service     = ['SMDVersion' => $smdVersion, 'serviceType' => $serviceType];
        $target      = $this->getTarget();
        $services    = $this->getServices();

        if (empty($services)) {
            return $service;
        }

        $service['methods'] = [];
        foreach ($services as $name => $svc) {
            $method = [
                'name'       => $name,
                'serviceURL' => $target,
            ];

            $params = [];
            foreach ($svc->getParams() as $param) {
                $params[] = [
                    'name' => array_key_exists('name', $param) ? $param['name'] : $param['type'],
                    'type' => $param['type'],
                ];
            }

            if (! empty($params)) {
                $method['parameters'] = $params;
            }

            $service['methods'][] = $method;
        }

        return $service;
    }

    /**
     * Cast to JSON.
     *
     * @return string
     */
    public function toJson()
    {
        return Json::encode($this->toArray());
    }

    /**
     * Cast to string (JSON)
     *
     * @return string
     */
    public function __toString()
    {
        return $this->toJson();
    }
}
